#!/usr/bin/env perl
#
# 2014/05/15 Gabriel Moreau <Gabriel.Moreau@univ-grenoble-alpes.fr>
# 2017/06/22 Gabriel Moreau - big update
# 2018/06/25 Gabriel Moreau - make velvice generic
#
# velvice.cgi
# Copyright (C) 2014-2018, LEGI UMR 5519 / CNRS UGA G-INP, Grenoble, France
#
# Need NagiosStatus http://exchange.nagios.org/directory/Addons/APIs/Perl/NagiosStatus-2Epm/details
# Possible command http://old.nagios.org/developerinfo/externalcommands/commandlist.php
#
# apt-get install perl-modules libnagios-object-perl libhtml-parser-perl liburi-encode-perl libcolor-calc-perl libyaml-syck-perl

use strict;
use warnings;
use version; our $VERSION = version->declare('0.6.2');

use CGI;
use HTML::Entities ();
use Nagios::StatusLog;
use URI::Encode qw(uri_encode uri_decode);
use Color::Calc ();
use YAML::Syck;

my $query           = CGI->new();
my $cgi_check       = uri_decode($query->param('check'));
my $cgi_script_name = $query->script_name();
my $cgi_uri         = $query->request_uri();

my $config = {};
$config = YAML::Syck::LoadFile('/etc/nagios3/velvice.yml') if -e '/etc/nagios3/velvice.yml';
$config->{'nagios-server'}                ||= {};
$config->{'nagios-server'}{'status-file'} ||= '/var/cache/nagios3/status.dat';
$config->{'nagios-server'}{'nagios-cmd'}  ||= '/var/lib/nagios3/rw/nagios.cmd';
$config->{'nagios-server'}{'portal-url'}  ||= $cgi_uri =~ s{/cgi-bin/}{/}r;
$config->{'nagios-server'}{'status-cgi'}  ||= "$cgi_uri/status.cgi";
$config->{'host-mapping'}                 ||= {};
$config->{'color-downtime'}               ||= {};
$config->{'color-downtime'}{'min'}        ||= 3;
$config->{'color-downtime'}{'max'}        ||= 50;
$config->{'color-downtime'}{'factor'}     ||= 0.7;
$config->{'remote-action'}                ||= {};

my $log = Nagios::StatusLog->new(
   Filename => $config->{'nagios-server'}{'status-file'},
   Version  => 3.0
   );

sub hostmapping {
   my $host = shift;

   return exists $config->{'host-mapping'}{$host} ? $config->{'host-mapping'}{$host} : $host;
   }

sub downtime {
   my ($time_change) = @_;

   my $now = time;
   return sprintf '%.1f', ($now - $time_change) / (60 * 3600);
   }

sub alertcolor {
   my ($color, $downtime) = @_;

   $downtime = $downtime - $config->{'color-downtime'}{'min'}; # same color first days
   $downtime = $config->{'color-downtime'}{'max'} if $downtime > $config->{'color-downtime'}{'max'}; #max 50 days for color
   $downtime =  0 if $downtime <  0;

   my $factor = ($downtime * $config->{'color-downtime'}{'factor'}) / $config->{'color-downtime'}{'max'};
   return Color::Calc::color_light_html($color, $factor);
   }

my %hostdown;
my @serviceproblems;
my %hostcount;
my @futurecheck;
HOST:
for my $host (sort $log->list_hosts()) {
   my $host_stat = $log->host($host);

   if ($host_stat->status eq 'DOWN') {TESTIF:{
      for my $srv ($log->list_services_on_host($host)) {
         last TESTIF if $log->service($host, $srv)->status eq 'OK' or $log->service($host, $srv)->status eq 'PENDING';
         }

      $hostdown{$host} = $host_stat;
      next HOST;
      }}

   for my $srv ($log->list_services_on_host($host)) {
      if ($log->service($host, $srv)->status ne 'OK') {
         push @serviceproblems, $log->service($host, $srv);
         $hostcount{$host}++;
         }
      }
   }

my $now = time;
my ($sec, $min, $hour, $mday, $mon, $year, $wday, $yday, $isdst) = localtime $now;
$year += 1900;
$mon++;
my $date = sprintf '%04i-%02i-%02i %02i:%02i', $year, $mon, $mday, $hour, $min;

my $htmlpage = <<"ENDH";
Content-Type: text/html

<!DOCTYPE HTML PUBLIC "-//W3C//DTD HTML 4.01 Transitional//EN" "http://www.w3.org/TR/html4/loose.dtd">
<html lang="en">
<head>
 <meta http-equiv="Content-Type" content="text/html; charset=UTF-8">
 <title>Nagios  Velvice</title>
</head>
<style type="text/css">
/* https://stackoverflow.com/questions/14920401/how-to-simulate-hfill-with-html-and-css */
h1 ul {
   display: flex;
   justify-content: space-between;
   }
h1 li {
   display: inline;
   }
td.bold {
   font-weight: bold;
   }
</style>
<body>
<h1>
 <ul>
   <li>Nagios Velvice Alert Panel : <a href="$config->{'nagios-server'}{'portal-url'}">Core Server</a></li>
   <li><small>(<a href="$cgi_script_name">UPDATE</a> - $date)</small></li>
 </ul>
</h1>
ENDH

my %service_name   = ();
my %service_status = ();
for my $srv (@serviceproblems) {
   $service_name{$srv->service_description}++;
   $service_status{$srv->status}++;
   }

if (scalar @serviceproblems == 0) {
   $htmlpage .= "<p>No alert to recheck.</p>\n";
   }
else {

   $htmlpage .= "<p>Alert to recheck - Level:\n";
   $htmlpage .= join ",\n",
      " <a href='$cgi_script_name?check=all'>ALL</a><small>(" . scalar(@serviceproblems) . ')</small>',
      map(" <a href='$cgi_script_name?check=" . lc(uri_encode($_)) . "'>$_</a>($service_status{$_})", sort keys %service_status);
   $htmlpage .= ".\n";
   $htmlpage .= " <br />\n";
   $htmlpage .= " Service:\n";
   $htmlpage .= join ",\n", map(" <a href='$cgi_script_name?check=" . lc(uri_encode($_)) . "'>$_</a><small>($service_name{$_})</small>", sort keys %service_name);
   $htmlpage .= ".\n";
   $htmlpage .= "</p>\n";

   my $nagios_cmd;
   open $nagios_cmd, '>>', $config->{'nagios-server'}{'nagios-cmd'} or die "Can't open file filename: $!";

   my %remote_sshdown = ();
   my %remote_db      = ();
   my $remote_flag;

   my $current_host  = '';
   $htmlpage .= "<table border=\"1\">\n";
   SERVICE_PROBLEMS:
   for my $srv (@serviceproblems) {
      my $hostname = $srv->host_name;
      my $service  = $srv->service_description;
      my $status   = $srv->status;
      my $downtime = downtime($srv->last_state_change);
      my $output   = HTML::Entities::encode($srv->plugin_output) =~ s/^[A-Z_\s]+?[:-]//r;

      my $color = $status eq 'CRITICAL' ? '#F88888' : '#FFFF00';
      $color = alertcolor($color, $downtime);
      $htmlpage .= " <tr style='background:$color;'>\n";
      if ($hostname ne $current_host) {
         $current_host = $hostname;
         $htmlpage .= "  <td rowspan='$hostcount{$hostname}' style='vertical-align:middle;'><a href=\"$config->{'nagios-server'}{'status-cgi'}?host=$hostname\">$hostname</a></td>\n";
         }

      my $bold;
      ACTION_STYLE:
      for my $act_name (keys %{$config->{'remote-action'}}) {
         my $act_regex = $config->{'remote-action'}{$act_name}{'regex'};
         $bold++ if $service =~ m/$act_regex/ and $config->{'remote-action'}{$act_name}{'style'} eq 'bold';
         }
      $htmlpage .= $bold ? '  <td class="bold">' : '  <td>';
      $htmlpage .= "$service</td>\n";

      $htmlpage .= "  <td>$status</td>\n";
      $htmlpage .= "  <td style='max-width:60%;'><small>$output";

      if (($cgi_check =~ m/all/i)
            or ($cgi_check =~ m/^$service$/i)
            or ($cgi_check =~ m/critical/i and $status eq 'CRITICAL')
            or ($cgi_check =~ m/warning/i  and $status eq 'WARNING')
            or ($cgi_check =~ m/pending/i  and $status eq 'PENDING')
            ) {
         $now++;
         my $interval = $srv->next_check() - $srv->last_check() || 300;
         $interval =  240 if $interval <  240;
         $interval = 3000 if $interval > 3000;
         my $future = $now + 20 + int(rand($interval - 20)); #5 * 60 = 300

         $htmlpage .= " -- <b>CHECK</b> [$now/" . ($future - $now) . "]";
         printf $nagios_cmd "[%lu] SCHEDULE_FORCED_SVC_CHECK;%s;%s;%lu\n", $now, $hostname, $service, $now;
         # delay future command
         push @futurecheck, sprintf "[%lu] SCHEDULE_FORCED_SVC_CHECK;%s;%s;%lu", $future, $hostname, $service, $future;
         }

      ACTION_PUSH_AND_DEPEND:
      for my $act_name (keys %{$config->{'remote-action'}}) {
         my $act_regex  = $config->{'remote-action'}{$act_name}{'regex'};
         my $act_status = $config->{'remote-action'}{$act_name}{'status'} || 'ALL';
         my $act_depend = $config->{'remote-action'}{$act_name}{'depend'} || 'SSH';

         if ($service =~ m/$act_regex/ and ($act_status eq 'ALL' or $status =~ m/$act_status/)) {
            $remote_db{$act_name} ||= [];
            push @{$remote_db{$act_name}}, $hostname;
            $remote_flag++;
            }

         # check depend service otherwise
         $remote_sshdown{$act_depend} ||= {};
         $remote_sshdown{$act_depend}->{$hostname}++ if $service =~ m/$act_depend/;
         }

      $htmlpage .= "</small></td>\n";
      $htmlpage .= "  <td style='text-align:right;'>$downtime days</td>\n";
      $htmlpage .= " </tr>\n";
      }

   $htmlpage .= "</table>\n";
   close $nagios_cmd;

   # host down
   if (%hostdown) {
      $htmlpage .= "<br />\n";
      $htmlpage .= "<table border='1'>\n";
      HOST_DOWN:
      for my $host (sort keys %hostdown) {
         my $host_stat = $hostdown{$host};
         my $hostname = $host_stat->host_name;
         my $downtime = downtime($host_stat->last_state_change);
         my $color = alertcolor('#F88888', $downtime);
         $htmlpage .= " <tr style='background:$color'>\n";
         $htmlpage .= "  <td><a href=\"$config->{'nagios-server'}{'status-cgi'}?host=$hostname\">$hostname</a></td>\n";
         my @host_service;
         for my $srv ($log->list_services_on_host($host)) {
            push @host_service, $log->service($host, $srv)->service_description;
            }
         $htmlpage .= "  <td><small>" . join(', ', @host_service) . "</small></td>\n";
         $htmlpage .= "  <td style='text-align:right;'>$downtime days</td>\n";
         $htmlpage .= " </tr>\n";
         }
      $htmlpage .= "</table>\n";
      }

   # remote action
   if ($remote_flag) {
      require Nagios::Object::Config;
      my $parser = Nagios::Object::Config->new();
      $parser->parse("/var/cache/nagios3/objects.cache");

      REMOTE_ACTION:
      for my $act_name (keys %remote_db) {
         my $act_depend = $config->{'remote-action'}{$act_name}{'depend'} || 'SSH';

         my @action = grep !exists $remote_sshdown{$act_depend}->{$_}, @{$remote_db{$act_name}};
         if (@action) {
            my $srv_title = $config->{'remote-action'}{$act_name}{'title'} || "Action: $act_name";
            $htmlpage .= "<h2>$srv_title</h2>\n";
            $htmlpage .= "<pre>\n";
            my $remote_action = $config->{'remote-action'}{$act_name}{'command'};
            $remote_action = $config->{'remote-action'}{$act_name}{'command-one'}
               if @action == 1 and exists $config->{'remote-action'}{$act_name}{'command-one'};
            my @hosts;
            for my $host (@action) {
               my $object = $parser->find_object("$host", "Nagios::Host");
               push @hosts, hostmapping($object->address =~ s/\..*$//r);
               }
            my $hosts_list = join ' ', @hosts;
            $htmlpage .= ' ' . $remote_action =~ s{\%m}{$hosts_list}r;
            $htmlpage .= "</pre>\n";
            }
         }
      }
   }

$htmlpage .= <<'ENDH';
</body>
</html>
ENDH

print $htmlpage;

# delayed future check
if (@futurecheck) {
   sleep 2;
   my $nagios_cmd;
   open $nagios_cmd, '>>', $config->{'nagios-server'}{'nagios-cmd'} or die "Can't open file filename: $!";
   print $nagios_cmd "$_\n" for @futurecheck;
   close $nagios_cmd;
   }

__END__


=head1 NAME

velvice.cgi - nagios velvice alert panel


=head1 DESCRIPTION

Nagios VELVICE is an acronym for "Nagios leVEL serVICE status".
Homepage: http://servforge.legi.grenoble-inp.fr/projects/soft-trokata/wiki/SoftWare/NagiosVelvice

=head1 AUTHORS

Written by Gabriel Moreau - Grenoble - France


=head1 LICENSE AND COPYRIGHT

Licence GNU GPL version 2 or later and Perl equivalent

Copyright (C) 2014-2018 Gabriel Moreau <Gabriel.Moreau(A)univ-grenoble-alpes.fr>.
