%'aver_stat': calculate field average, used with series.fig
%------------------------------------------------------------------------
% function GUI_input=aver_stat(Param)
%
%%%%%%%%%%% GENERAL TO ALL SERIES ACTION FCTS %%%%%%%%%%%%%%%%%%%%%%%%%%%
%OUTPUT
% GUI_input=list of options in the GUI series.fig needed for the function
%
%INPUT:
% In run mode, the input parameters are given as a Matlab structure Param copied from the GUI series.
% In batch mode, Param is the name of the corresponding xml file containing the same information
% In the absence of input (as activated when the current Action is selected
% in series), the function ouput GUI_input set the activation of the needed GUI elements
%
% Param contains the elements:(use the menu bar command 'export/GUI config' in series to see the current structure Param)
%    .InputTable: cell of input file names, (several lines for multiple input)
%                      each line decomposed as {RootPath,SubDir,Rootfile,NomType,Extension}
%    .OutputSubDir: name of the subdirectory for data outputs
%    .OutputDir: directory for data outputs, including path
%    .Action: .ActionName: name of the current activated function
%             .ActionPath:   path of the current activated function
%    .IndexRange: set the file or frame indices on which the action must be performed
%    .FieldTransform: .TransformName: name of the selected transform function
%                     .TransformPath:   path  of the selected transform function
%                     .TransformHandle: corresponding function handle
%    .InputFields: sub structure describing the input fields withfields
%              .FieldName: name of the field
%              .VelType: velocity type
%              .FieldName_1: name of the second field in case of two input series
%              .VelType_1: velocity type of the second field in case of two input series
%    .ProjObject: %sub structure describing a projection object (read from ancillary GUI set_object)
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function GUI_input=aver_stat(Param)

%% set the input elements needed on the GUI series when the action is selected in the menu ActionName
if ~exist('Param','var') % case with no input parameter 
    GUI_input={'NbViewMax';2;...% max nbre of input file series (default='' , no limitation)
        'AllowInputSort';'off';...% allow alphabetic sorting of the list of input files (options 'off'/'on', 'off' by default)
        'NbSlice';'on'; ...%nbre of slices ('off' by default)
        'VelType';'two';...% menu for selecting the velocity type (options 'off'/'one'/'two',  'off' by default)
        'FieldName';'two';...% menu for selecting the field (s) in the input file(options 'off'/'one'/'two', 'off' by default)
        'FieldTransform'; 'on';...%can use a transform function
        'ProjObject';'on';...%can use projection object(option 'off'/'on',
        'Mask';'off';...%can use mask option   (option 'off'/'on', 'off' by default)
        'OutputDirExt';'.stat';...%set the output dir extension
               ''};
        return
end

%%%%%%%%%%%% STANDARD PART (DO NOT EDIT) %%%%%%%%%%%%
%% get input parameters, file names and indices
% BATCH  case: read the xml file for batch case
if ischar(Param) && ~isempty(find(regexp('Param','.xml$')))
    Param=xml2struct(Param);
    checkrun=0;
% RUN case: parameters introduced as the input structure Param  
else 
    hseries=guidata(Param.hseries);%handles of the GUI series
    WaitbarPos=get(hseries.waitbar_frame,'Position');%position of the waitbar on the GUI series
    checkrun=1; % indicate the RUN option is used
end
% get the set of input file names (cell array filecell), and the lists of
% input file or frame indices i1_series,i2_series,j1_series,j2_series
[filecell,i1_series,i2_series,j1_series,j2_series]=get_file_series(Param);
% filecell{iview,fileindex}: cell array representing the list of file names
%        iview: line in the table corresponding to a given file series
%        fileindex: file index within  the file series, 
% i1_series(iview,ref_j,ref_i)... are the corresponding arrays of indices i1,i2,j1,j2, depending on the input line iview and the two reference indices ref_i,ref_j 
% i1_series(iview,fileindex) expresses the same indices as a 1D array in file indices
% set of frame indices used for movie or multimage input 
if ~isempty(j1_series)
    frame_index=j1_series;
else
    frame_index=i1_series;
end

%% root input file(s) and type
RootPath=Param.InputTable(:,1);
RootFile=Param.InputTable(:,3);
SubDir=Param.InputTable(:,2);
NomType=Param.InputTable(:,4);
FileExt=Param.InputTable(:,5);

% numbers of slices and file indices
NbSlice=1;%default
if isfield(Param.IndexRange,'NbSlice')
    NbSlice=Param.IndexRange.NbSlice;
end
nbview=size(i1_series,1);%number of input file series (lines in InputTable)
nbfield_j=size(i1_series,2); %nb of consecutive fields at each level(burst
nbfield=nbfield_j*size(i1_series,3); %total number of files or frames
nbfield_i=floor(nbfield/NbSlice);%total number of i indexes (adjusted to an integer number of slices)
nbfield=nbfield_i*nbfield_j; %total number of fields after adjustement

%determine the file type on each line from the first input file 
ImageTypeOptions={'image','multimage','mmreader','video'};
NcTypeOptions={'netcdf','civx','civdata'};
for iview=1:nbview
    [FileType{iview},FileInfo{iview},Object{iview}]=get_file_type(filecell{iview,1});
    CheckImage{iview}=~isempty(find(strcmp(FileType{iview},ImageTypeOptions)));% =1 for images
    CheckNc{iview}=~isempty(find(strcmp(FileType{iview},NcTypeOptions)));% =1 for netcdf files
end

%% Calibration data and timing: read the ImaDoc files
mode=''; %default
timecell={};
itime=0;
NbSlice_calib={};
for iview=1:nbview%Loop on views
    XmlData{iview}=[];%default
    filebase{iview}=fullfile(RootPath{iview},RootFile{iview});
    if exist([filebase{iview} '.xml'],'file')
        [XmlData{iview},error]=imadoc2struct([filebase{iview} '.xml']); 
        if isfield(XmlData{iview},'Time')
            itime=itime+1;
            timecell{itime}=XmlData{iview}.Time;
        end
        if isfield(XmlData{iview},'GeometryCalib') && isfield(XmlData{iview}.GeometryCalib,'SliceCoord')
            NbSlice_calib{iview}=size(XmlData{iview}.GeometryCalib.SliceCoord,1);%nbre of slices for Zindex in phys transform
            if ~isequal(NbSlice_calib{iview},NbSlice_calib{1})
                msgbox_uvmat('WARNING','inconsistent number of Z indices for the two field series');
            end
        end 
    elseif exist([filebase{iview} '.civ'],'file')
        [error,time,TimeUnit,mode,npx,npy,pxcmx,pxcmy]=read_imatext([filebase{iview} '.civ']);
        itime=itime+1;
        timecell{itime}=time;
        XmlData{iview}.Time=time;
        GeometryCalib.R=[pxcmx 0 0; 0 pxcmy 0;0 0 0];
        GeometryCalib.Tx=0;
        GeometryCalib.Ty=0;
        GeometryCalib.Tz=1;
        GeometryCalib.dpx=1;
        GeometryCalib.dpy=1;
        GeometryCalib.sx=1;
        GeometryCalib.Cx=0;
        GeometryCalib.Cy=0;
        GeometryCalib.f=1;
        GeometryCalib.kappa1=0;
        GeometryCalib.CoordUnit='cm';
        XmlData{iview}.GeometryCalib=GeometryCalib;
        if error==1
            msgbox_uvmat('WARNING','inconsistent number of fields in the .civ file');
        end
    end
end

%% check coincidence in time for several input file series
multitime=0;
if isempty(timecell)
    time=[];
elseif length(timecell)==1
    time=timecell{1};
elseif length(timecell)>1
    multitime=1;
    for icell=1:length(timecell)
        if ~isequal(size(timecell{icell}),size(timecell{1}))
            msgbox_uvmat('WARNING','inconsistent time array dimensions in ImaDoc fields, the time for the first series is used')
            time=timecell{1};
            multitime=0;
            break
        end
    end
end
if multitime
    for icell=1:length(timecell)
        time(icell,:,:)=timecell{icell};
    end
    diff_time=max(max(diff(time)));
    if diff_time>0
        msgbox_uvmat('WARNING',['times of series differ by (max) ' num2str(diff_time)])
    end   
end
if size(time,2) < i2_series(1,end) || size(time,3) < j2_series(1,end)% time array absent or too short in ImaDoc xml file' 
    time=[];
end

%% coordinate transform or other user defined transform
transform_fct='';%default
if isfield(Param,'FieldTransform')&&isfield(Param.FieldTransform,'TransformHandle')
    transform_fct=Param.FieldTransform.TransformHandle;
end
%%%%%%%%%%%% END STANDARD PART  %%%%%%%%%%%%
 % EDIT FROM HERE

%% check the validity of  input file types
if CheckImage{1}
    FileExtOut='.png'; % write result as .png images for image inputs
elseif CheckNc{1}
    FileExtOut='.nc';% write result as .nc files for netcdf inputs
else 
    msgbox_uvmat('ERROR',['invalid file type input ' FileType{1}])
    return
end
if nbview==2 && ~isequal(CheckImage{1},CheckImage{2})
        msgbox_uvmat('ERROR','input must be two image series or two netcdf file series')
    return
end
NomTypeOut='_1-2_1';% output file index will indicate the first and last ref index in the series
if NbSlice~=nbfield_j
    answer=msgbox_uvmat('INPUT_Y-N',['will not average slice by slice: for so cancel and set NbSlice= ' num2str(nbfield_j)]);
    if ~strcmp(answer,'Yes')
        return
    end
end

%% Set field names and velocity types
InputFields{1}=[];%default (case of images)
if isfield(Param,'InputFields')
    InputFields{1}=Param.InputFields;
end
if nbview==2
    InputFields{2}=[];%default (case of images)
    if isfield(Param,'InputFields')
        InputFields{2}=Param.InputFields{1};%default
        if isfield(Param.InputFields,'FieldName_1')
            InputFields{2}.FieldName=Param.InputFields.FieldName_1;
            if isfield(Param.InputFields,'VelType_1')
                InputFields{2}.VelType=Param.InputFields.VelType_1;
            end
        end
    end
end

%% Initiate output fields
%initiate the output structure as a copy of the first input one (reproduce fields)
[DataOut,ParamOut,errormsg] = read_field(filecell{1,1},FileType{1},InputFields{1},1);
if ~isempty(errormsg)
    msgbox_uvmat('ERROR',['error reading ' filecell{1,1} ': ' errormsg])
    return
end
time_1=[];
if isfield(DataOut,'Time')
    time_1=DataOut.Time(1);
end
if CheckNc{iview}
    if isempty(strcmp('Conventions',DataOut.ListGlobalAttribute))
        DataOut.ListGlobalAttribute=['Conventions' DataOut.ListGlobalAttribute];
    end
    DataOut.Conventions='uvmat';
    DataOut.ListGlobalAttribute=[DataOut.ListGlobalAttribute {Param.Action}];
    ActionKey='Action';
    while isfield(DataOut,ActionKey)
        ActionKey=[ActionKey '_1'];
    end
    DataOut.(ActionKey)=Param.Action;
    DataOut.ListGlobalAttribute=[DataOut.ListGlobalAttribute {ActionKey}];
    if isfield(DataOut,'Time')
        DataOut.ListGlobalAttribute=[DataOut.ListGlobalAttribute {'Time','Time_end'}];
    end
end

%% MAIN LOOP ON SLICES
%%%%%%%%%%%%% STANDARD PART (DO NOT EDIT) %%%%%%%%%%%%
for i_slice=1:NbSlice
    index_slice=i_slice:NbSlice:nbfield;% select file indices of the slice
    nbfiles=0;
    nbmissing=0;
    
   %initiate result fields
   for ivar=1:length(DataOut.ListVarName)
       DataOut.(DataOut.ListVarName{ivar})=0; % initialise all fields to zero
   end

    %%%%%%%%%%%%%%%% loop on field indices %%%%%%%%%%%%%%%%
    for index=index_slice
        if checkrun
            update_waitbar(hseries.waitbar_frame,WaitbarPos,index/(nbfield))
            stopstate=get(hseries.RUN,'BusyAction');
        else
            stopstate='queue';
        end
        
        %%%%%%%%%%%%%%%% loop on views (input lines) %%%%%%%%%%%%%%%%
        for iview=1:nbview
            % reading input file(s)
            [Data{iview},ParamOut,errormsg] = read_field(filecell{iview,index},FileType{iview},InputFields{iview},frame_index(iview,index));
            if ~isempty(errormsg)
                errormsg=['error of input reading: ' errormsg];
                break
            end
            if ~isempty(NbSlice_calib)
                Data{iview}.ZIndex=mod(i1_series(iview,index)-1,NbSlice_calib{1})+1;%Zindex for phys transform
            end
        end
        Field=[]; % initiate the current input field structure
        %%%%%%%%%%%%%%%% end loop on views (input lines) %%%%%%%%%%%%%%%%
        %%%%%%%%%%%% END STANDARD PART  %%%%%%%%%%%%
        % EDIT FROM HERE

        if isempty(errormsg)     
            % coordinate transform (or other user defined transform)
            if ~isempty(transform_fct)
                if nbview==2
                    [Data{1},Data{2}]=transform_fct(Data{1},XmlData{1},Data{2},XmlData{2});
                    if isempty(Data{2})
                        Data(2)=[];
                    end
                else
                    Data{1}=transform_fct(Data{1},XmlData{1});
                end
            end
            
            % field calculation (vort, div...)
            if strcmp(FileType{1},'civx')||strcmp(FileType{1},'civ')
                Data{1}=calc_field(InputFields{1}.FieldName,Data{1});%calculate field (vort..)
            end
            
            % field substration (for two input file series)
            if length(Data)==2
                if strcmp(FileType{2},'civx')||strcmp(FileType{2},'civ')
                    Data{2}=calc_field(InputFields{2}.FieldName,Data{2});%calculate field (vort..)
                end
                [Field,errormsg]=sub_field(Data{1},Data{2}); %substract the two fields
                if ~isempty(errormsg)
                    msgbox_uvmat('ERROR',['error in aver_stat/sub_field:' errormsg])
                    return
                end
            else
                Field=Data{1};
            end
            if Param.CheckObject
                [Field,errormsg]=proj_field(Field,ProjObject);
                if ~isempty(errormsg)
                    msgbox_uvmat('ERROR',['error in aver_stat/proj_field:' errormsg])
                    return
                end
            end
            nbfiles=nbfiles+1;
            
            %%%%%%%%%%%% MAIN RUNNING OPERATIONS  %%%%%%%%%%%%
            %update sum
            for ivar=1:length(Field.ListVarName)
                VarName=Field.ListVarName{ivar};
                sizmean=size(DataOut.(VarName));
                siz=size(Field.(VarName));
                if ~isequal(DataOut.(VarName),0)&& ~isequal(siz,sizmean)
                    msgbox_uvmat('ERROR',['unequal size of input field ' VarName ', need to project  on a grid'])
                    return
                else
                    DataOut.(VarName)=DataOut.(VarName)+ double(Field.(VarName)); % update the sum
                end
            end
            %%%%%%%%%%%%   END MAIN RUNNING OPERATIONS  %%%%%%%%%%%%
        else
            display(errormsg)  
        end
    end
    %%%%%%%%%%%%%%%% end loop on field indices %%%%%%%%%%%%%%%%
    
    for ivar=1:length(Field.ListVarName)
        VarName=Field.ListVarName{ivar};
        DataOut.(VarName)=DataOut.(VarName)/nbfiles; % normalize the mean
    end
    if nbmissing~=0
        msgbox_uvmat('WARNING',[num2str(nbmissing) ' input files are missing or skipted'])
    end
    if isempty(time) % time read from files  prevails
        if isfield(Field,'Time')
            time_end=Field.Time(1);%last time read
            if ~isempty(time_1)
                DataOut.Time=time_1;
                DataOut.Time_end=time_end;
            end
        end
    else  % time from ImaDoc prevails
        DataOut.Time=time(1,i1_series(1,1),j1_series(1,1));
        DataOut.Time_end=time(end,i1_series(end,end),j1_series(end,end));
    end
    
    %writing the result file
    OutputFile=fullfile_uvmat(RootPath{1},Param.OutputSubDir,RootFile{1},FileExtOut,NomTypeOut,i1_series(1,1),i1_series(1,end),i_slice,[]);
    if CheckImage{1} %case of images
        if isequal(FileInfo{1}.BitDepth,16)||(numel(FileInfo)==2 &&isequal(FileInfo{2}.BitDepth,16))
            DataOut.A=uint16(DataOut.A);
            imwrite(DataOut.A,OutputFile,'BitDepth',16); % case of 16 bit images
        else
            DataOut.A=uint8(DataOut.A);
            imwrite(DataOut.A,OutputFile,'BitDepth',8); % case of 16 bit images
        end
        display([OutputFile ' written']);
    else %case of netcdf input file , determine global attributes
        errormsg=struct2nc(OutputFile,DataOut); %save result file
        if isempty(errormsg)
            display([OutputFile ' written']);
        else
            msgbox_uvmat('ERROR',['error in writting result file: ' errormsg])
            display(errormsg)
        end
    end  % end averaging  loop
end
%%%%%%%%%%%%%%%% end loop on slices %%%%%%%%%%%%%%%%

%% reproduce ImaDoc/GeometryCalib for image series
% if isfield(XmlData{1},'GeometryCalib') && ~isempty(XmlData{1}.GeometryCalib) 
%     [tild,RootFile]=fileparts(filebase_out);
%     outputxml=fullfile(pathdir,[RootFile '.xml']);
%     errormsg=update_imadoc(XmlData{1}.GeometryCalib,outputxml);% introduce the calibration data in the xml file
%     if strcmp(errormsg,'')
%         display(['GeometryCalib transferred to ' outputxml])
%     else
%         msgbox_uvmat('ERROR',errormsg);
%     end
% end

%% open the result file with uvmat
hget_field=findobj(allchild(0),'name','get_field');%find the get_field... GUI
delete(hget_field)
uvmat(OutputFile)% open the last result file with uvmat